<?php
/**
 * Theme Info Page
 *
 * @package Divine Word
 */

namespace Divine_Word;

use const DAY_IN_SECONDS;

/**
 * Exit if accessed directly.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Divine_Word_Theme_Notice();

class Divine_Word_Theme_Notice {

	/** @var \WP_Theme */
	private $divine_word_theme;

	private $divine_word_url = 'https://www.themescarts.com/';

	/**
	 * Class construct.
	 */
	public function __construct() {
		$this->divine_word_theme = wp_get_theme();

		add_action( 'init', array( $this, 'handle_dismiss_notice' ) );

		if ( ! \get_transient( 'divine_word_notice_dismissed' ) ) {
			add_action( 'admin_notices', array( $this, 'divine_word_render_notice' ) );
		}

		add_action( 'switch_theme', array( $this, 'show_notice' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'divine_word_enqueue_notice_assets' ) );
	}

	/**
	 * Delete notice on theme switch.
	 *
	 * @return void
	 */
	public function show_notice() {
		\delete_transient( 'divine_word_notice_dismissed' );
	}

	/**
	 * Enqueue admin styles and scripts.
	 */
	public function divine_word_enqueue_notice_assets() {
		wp_enqueue_style(
			'divine-word-theme-notice-css',
			get_template_directory_uri() . '/inc/divine-word-theme-info-page/css/theme-details.css',
			array(),
			'1.0.0'
		);

		wp_enqueue_script(
			'divine-word-theme-notice-js',
			get_template_directory_uri() . '/inc/divine-word-theme-info-page/js/theme-details.js',
			array( 'jquery' ),
			'1.0.0',
			true
		);

		// Pass dynamic URL to JS
		wp_localize_script( 'divine-word-theme-notice-js', 'DivineWordTheme', array(
			'admin_url' => esc_url( admin_url( 'admin.php?page=themescarts' ) ),
		));
	}

	/**
	 * Render the admin notice.
	 */
	public function divine_word_render_notice() {
		?>
		<div id="divine-word-theme-notice" class="notice notice-info is-dismissible">
			<div class="divine-word-content-wrap">
				<div class="divine-word-notice-left">
					<?php
					$this->divine_word_render_title();
					$this->divine_word_render_content();
					$this->divine_word_render_actions();
					?>
				</div>
				<div class="divine-word-notice-right">
					<img src="<?php echo esc_url( get_template_directory_uri() . '/screenshot.png' ); ?>" alt="<?php esc_attr_e( 'Theme Notice Image', 'divine-word' ); ?>">
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render title.
	 */
	protected function divine_word_render_title() {
		?>
		<h2>
			<?php
			printf(
				// translators: %s is the theme name
				esc_html__( 'Thank you for installing %s!', 'divine-word' ),
				'<span>' . esc_html( $this->divine_word_theme->get( 'Name' ) ) . '</span>'
			);
			?>
		</h2>
		<?php
	}

	/**
	 * Render content.
	 */
	protected function divine_word_render_content() {
		$divine_word_link = '<a href="' . esc_url( $this->divine_word_url ) . '" target="_blank">' . esc_html__( 'ThemesCarts', 'divine-word' ) . '</a>';

		$divine_word_text = sprintf(
			/* translators: %1$s: Author Name, %2$s: Link */
			esc_html__( 'Unlock the full potential of your new store with %1$s! Get started today by visiting %2$s to explore a wide range of ready-to-use patterns and demo templates, designed to enhance your online shopping experience.', 'divine-word' ),
			esc_html__( 'ThemesCarts', 'divine-word' ),
			$divine_word_link
		);

		echo wp_kses_post( wpautop( $divine_word_text ) );
	}

	/**
	 * Render action buttons.
	 */
	protected function divine_word_render_actions() {
		$divine_word_more_info_url = admin_url( 'themes.php?page=divine-word-theme-info-page' );
		?>
		<div class="notice-actions">
			<a href="<?php echo esc_url( $divine_word_more_info_url ); ?>" id="btn-install-activate">
				<?php esc_html_e( 'Click Here For Theme Info', 'divine-word' ); ?>
			</a>
			<form class="divine-word-notice-dismiss-form" method="post">
				<button type="submit" name="notice-dismiss" value="true" id="btn-dismiss">
					<?php esc_html_e( 'Dismiss', 'divine-word' ); ?>
				</button>
			</form>
		</div>
		<?php
	}

	/**
	 * Handle dismiss action.
	 */
	public function handle_dismiss_notice() {
		if ( isset( $_POST['notice-dismiss'] ) ) {
			set_transient( 'divine_word_notice_dismissed', true, DAY_IN_SECONDS * 3 );
			wp_safe_redirect( esc_url_raw( $_SERVER['REQUEST_URI'] ) );
			exit;
		}
	}
}
?>