<?php
/**
 * Real Estate Management functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Real Estate Management
 */

include get_theme_file_path( 'vendor/wptrt/autoload/src/Real_Estate_Management_Loader.php' );

$Real_Estate_Management_Loader = new \WPTRT\Autoload\Real_Estate_Management_Loader();

$Real_Estate_Management_Loader->real_estate_management_add( 'WPTRT\\Customize\\Section', get_theme_file_path( 'vendor/wptrt/customize-section-button/src' ) );

$Real_Estate_Management_Loader->real_estate_management_register();

if ( ! function_exists( 'real_estate_management_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function real_estate_management_setup() {

		load_theme_textdomain( 'real-estate-management', get_template_directory() . '/languages' );

		/*
		 * Enable support for Post Formats.
		 *
		 * See: https://codex.wordpress.org/Post_Formats
		*/
		add_theme_support( 'post-formats', array('image','video','gallery','audio',) );
		
		add_theme_support( 'woocommerce' );
		add_theme_support( "responsive-embeds" );
		add_theme_support( "align-wide" );
		add_theme_support( "wp-block-styles" );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );

        add_image_size('real-estate-management-featured-header-image', 2000, 660, true);

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus( array(
            'primary' => esc_html__( 'Primary','real-estate-management' ),
	        'footer'=> esc_html__( 'Footer Menu','real-estate-management' ),
        ) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'real_estate_management_custom_background_args', array(
			'default-color' => 'f7ebe5',
			'default-image' => '',
		) ) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support( 'custom-logo', array(
			'height'      => 50,
			'width'       => 50,
			'flex-width'  => true,
		) );

		add_editor_style( array( '/editor-style.css' ) );
		add_action('wp_ajax_real_estate_management_dismissable_notice', 'real_estate_management_dismissable_notice');
	}
endif;
add_action( 'after_setup_theme', 'real_estate_management_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function real_estate_management_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'real_estate_management_content_width', 1170 );
}
add_action( 'after_setup_theme', 'real_estate_management_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function real_estate_management_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'real-estate-management' ),
		'id'            => 'sidebar',
		'description'   => esc_html__( 'Add widgets here.', 'real-estate-management' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h5 class="widget-title">',
		'after_title'   => '</h5>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Woocommerce Single Product Page Sidebar', 'real-estate-management' ),
		'id'            => 'woocommerce-single-product-page-sidebar',
		'description'   => '',
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h5 class="widget-title">',
		'after_title'   => '</h5>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Woocommerce Shop Page Sidebar', 'real-estate-management' ),
		'id'            => 'woocommerce-shop-page-sidebar',
		'description'   => '',
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h5 class="widget-title">',
		'after_title'   => '</h5>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer Column 1', 'real-estate-management' ),
		'id'            => 'real-estate-management-footer1',
		'description'   => '',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h5 class="footer-column-widget-title">',
		'after_title'   => '</h5>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer Column 2', 'real-estate-management' ),
		'id'            => 'real-estate-management-footer2',
		'description'   => '',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h5 class="footer-column-widget-title">',
		'after_title'   => '</h5>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer Column 3', 'real-estate-management' ),
		'id'            => 'real-estate-management-footer3',
		'description'   => '',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h5 class="footer-column-widget-title">',
		'after_title'   => '</h5>',
	) );
}
add_action( 'widgets_init', 'real_estate_management_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function real_estate_management_scripts() {

	require_once get_theme_file_path( 'inc/wptt-webfont-loader.php' );

	wp_enqueue_style(
		'noto',
		wptt_get_webfont_url( 'https://fonts.googleapis.com/css2?family=Noto+Sans:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&display=swap' ),
		array(),
		'1.0'
	);

	wp_enqueue_style( 'real-estate-management-block-editor-style', get_theme_file_uri('/assets/css/block-editor-style.css') );

	// load bootstrap css
    wp_enqueue_style( 'bootstrap-css', get_template_directory_uri() . '/assets/css/bootstrap.css');

    wp_enqueue_style( 'owl.carousel-css', get_template_directory_uri() . '/assets/css/owl.carousel.css');

	wp_enqueue_style( 'real-estate-management-style', get_stylesheet_uri() );
	require get_parent_theme_file_path( '/custom-option.php' );
	wp_add_inline_style( 'real-estate-management-style',$real_estate_management_theme_css );

	wp_style_add_data('real-estate-management-basic-style', 'rtl', 'replace');

	// fontawesome
	wp_enqueue_style( 'fontawesome-style', get_template_directory_uri() .'/assets/css/fontawesome/css/all.css' );

    wp_enqueue_script('real-estate-management-theme-js', get_template_directory_uri() . '/assets/js/theme-script.js', array('jquery'), '', true );

    wp_enqueue_script('owl.carousel-js', get_template_directory_uri() . '/assets/js/owl.carousel.js', array('jquery'), '', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'real_estate_management_scripts' );


/**
 * Enqueue Preloader.
 */
function real_estate_management_preloader() {

  $real_estate_management_theme_color_css = '';
  $real_estate_management_preloader_bg_color = get_theme_mod('real_estate_management_preloader_bg_color');
  $real_estate_management_preloader_dot_1_color = get_theme_mod('real_estate_management_preloader_dot_1_color');
  $real_estate_management_preloader_dot_2_color = get_theme_mod('real_estate_management_preloader_dot_2_color');
  $real_estate_management_preloader2_dot_color = get_theme_mod('real_estate_management_preloader2_dot_color');
  $real_estate_management_logo_max_height = get_theme_mod('real_estate_management_logo_max_height');

	if(get_theme_mod('real_estate_management_logo_max_height') == '') {
		$real_estate_management_logo_max_height = '24';
	}
	if(get_theme_mod('real_estate_management_preloader_dot_1_color') == '') {
		$real_estate_management_preloader_dot_1_color = '#fff';
	}
	if(get_theme_mod('real_estate_management_preloader_dot_2_color') == '') {
		$real_estate_management_preloader_dot_2_color = '#ff6f2f';
	}
	$real_estate_management_theme_color_css = '
	.custom-logo-link img{
			max-height: '.esc_attr($real_estate_management_logo_max_height).'px;
		 }
		.loading, .loading2{
			background-color: '.esc_attr($real_estate_management_preloader_bg_color).';
		 }
		 @keyframes loading {
		  0%,
		  100% {
		  	transform: translatey(-2.5rem);
		    background-color: '.esc_attr($real_estate_management_preloader_dot_1_color).';
		  }
		  50% {
		  	transform: translatey(2.5rem);
		    background-color: '.esc_attr($real_estate_management_preloader_dot_2_color).';
		  }
		}
		.load hr {
			background-color: '.esc_attr($real_estate_management_preloader2_dot_color).';
		}
	';
    wp_add_inline_style( 'real-estate-management-style',$real_estate_management_theme_color_css );

}
add_action( 'wp_enqueue_scripts', 'real_estate_management_preloader' );

function real_estate_management_files_setup() {

	/**
	 * Custom template tags for this theme.
	 */
	require get_template_directory() . '/inc/template-tags.php';

	/**
	 * Implement the Custom Header feature.
	 */
	require get_template_directory() . '/inc/custom-header.php';

	/**
	 * Functions which enhance the theme by hooking into WordPress.
	 */
	require get_template_directory() . '/inc/template-functions.php';

	/**
	 * Customizer additions.
	 */
	require_once get_template_directory() . '/inc/customizer-seperator.php';

	require get_template_directory() . '/inc/customizer.php';

	/**
	 * Meta Feild
	 */
	require get_template_directory() . '/inc/popular-property-meta.php';

	require get_template_directory() . '/inc/tgm.php';


	if ( ! defined( 'REAL_ESTATE_MANAGEMENT_CONTACT_SUPPORT' ) ) {
		define('REAL_ESTATE_MANAGEMENT_CONTACT_SUPPORT',__('https://wordpress.org/support/theme/real-estate-management','real-estate-management'));
	}
	if ( ! defined( 'REAL_ESTATE_MANAGEMENT_REVIEW' ) ) {
		define('REAL_ESTATE_MANAGEMENT_REVIEW',__('https://wordpress.org/support/theme/real-estate-management/reviews/#new-post','real-estate-management'));
	}
	if ( ! defined( 'REAL_ESTATE_MANAGEMENT_LIVE_DEMO' ) ) {
		define('REAL_ESTATE_MANAGEMENT_LIVE_DEMO',__('https://demo.themagnifico.net/real-estate-management/','real-estate-management'));
	}
	if ( ! defined( 'REAL_ESTATE_MANAGEMENT_GET_PREMIUM_PRO' ) ) {
		define('REAL_ESTATE_MANAGEMENT_GET_PREMIUM_PRO',__('https://www.themagnifico.net/products/property-management-wordpress-theme','real-estate-management'));
	}
	if ( ! defined( 'REAL_ESTATE_MANAGEMENT_PRO_DOC' ) ) {
		define('REAL_ESTATE_MANAGEMENT_PRO_DOC',__('https://demo.themagnifico.net/eard/wathiqa/real-estate-management-pro-doc/','real-estate-management'));
	}
	if ( ! defined( 'REAL_ESTATE_MANAGEMENT_FREE_DOC' ) ) {
		define('REAL_ESTATE_MANAGEMENT_FREE_DOC',__('https://demo.themagnifico.net/eard/wathiqa/real-estate-management-free-doc/','real-estate-management'));
	}


}

add_action( 'after_setup_theme', 'real_estate_management_files_setup' );

/** * Posts pagination. */
if ( ! function_exists( 'real_estate_management_blog_posts_pagination' ) ) {
	function real_estate_management_blog_posts_pagination() {
		$pagination_type = get_theme_mod( 'real_estate_management_blog_pagination_type', 'blog-nav-numbers' );
		if ( $pagination_type == 'blog-nav-numbers' ) {
			the_posts_pagination();
		} else {
			the_posts_navigation();
		}
	}
}

/*dropdown page sanitization*/
function real_estate_management_sanitize_dropdown_pages( $page_id, $setting ) {
	$page_id = absint( $page_id );
	return ( 'publish' == get_post_status( $page_id ) ? $page_id : $setting->default );
}

function real_estate_management_sanitize_select( $input, $setting ){
    $input = sanitize_key($input);
    $choices = $setting->manager->get_control( $setting->id )->choices;
    return ( array_key_exists( $input, $choices ) ? $input : $setting->default );
}

function real_estate_management_sanitize_checkbox( $input ) {
	// Boolean check
	return ( ( isset( $input ) && true == $input ) ? true : false );
}

function real_estate_management_preloader1(){
	if(get_theme_mod('real_estate_management_preloader_type', 'Preloader 1') == 'Preloader 1' ) {
		return true;
	}
	return false;
}

function real_estate_management_preloader2(){
	if(get_theme_mod('real_estate_management_preloader_type', 'Preloader 1') == 'Preloader 2' ) {
		return true;
	}
	return false;
}

function real_estate_management_sanitize_number_absint( $number, $setting ) {
	// Ensure $number is an absolute integer (whole number, zero or greater).
	$number = absint( $number );

	// If the input is an absolute integer, return it; otherwise, return the default
	return ( $number ? $number : $setting->default );
}

function real_estate_management_sanitize_phone_number( $phone ) {
	return preg_replace( '/[^\d+]/', '', $phone );
}

/*radio button sanitization*/
function real_estate_management_sanitize_choices( $input, $setting ) {
    global $wp_customize;
    $control = $wp_customize->get_control( $setting->id );
    if ( array_key_exists( $input, $control->choices ) ) {
        return $input;
    } else {
        return $setting->default;
    }
}

function real_estate_management_sanitize_number_range( $number, $setting ) {
	
	// Ensure input is an absolute integer.
	$number = absint( $number );
	
	// Get the input attributes associated with the setting.
	$atts = $setting->manager->get_control( $setting->id )->input_attrs;
	
	// Get minimum number in the range.
	$min = ( isset( $atts['min'] ) ? $atts['min'] : $number );
	
	// Get maximum number in the range.
	$max = ( isset( $atts['max'] ) ? $atts['max'] : $number );
	
	// Get step.
	$step = ( isset( $atts['step'] ) ? $atts['step'] : 1 );
	
	// If the number is within the valid range, return it; otherwise, return the default
	return ( $min <= $number && $number <= $max && is_int( $number / $step ) ? $number : $setting->default );
}

// Change number or products per row to 3
add_filter('loop_shop_columns', 'real_estate_management_loop_columns');
if (!function_exists('real_estate_management_loop_columns')) {
	function real_estate_management_loop_columns() {
		$columns = get_theme_mod( 'real_estate_management_products_per_row', 3 );
		return $columns; // 3 products per row
	}
}

/**
 * Get CSS
 */

function real_estate_management_getpage_css($hook) {
	wp_register_script( 'admin-notice-script', get_template_directory_uri() . '/inc/admin/js/admin-notice-script.js', array( 'jquery' ) );
    wp_localize_script('admin-notice-script','real_estate_management',
		array('admin_ajax'	=>	admin_url('admin-ajax.php'),'wpnonce'  =>	wp_create_nonce('real_estate_management_dismissed_notice_nonce')
		)
	);
	wp_enqueue_script('admin-notice-script');

    wp_localize_script( 'admin-notice-script', 'real_estate_management_ajax_object',
        array( 'ajax_url' => admin_url( 'admin-ajax.php' ) )
    );
	if ( 'appearance_page_real-estate-management-info' != $hook ) {
		return;
	}
}
add_action( 'admin_enqueue_scripts', 'real_estate_management_getpage_css' );

//Admin Notice For Getstart
function real_estate_management_ajax_notice_handler() {
    if ( isset( $_POST['type'] ) ) {
        $type = sanitize_text_field( wp_unslash( $_POST['type'] ) );
        update_option( 'dismissed-' . $type, TRUE );
    }
}

function real_estate_management_deprecated_hook_admin_notice() {

    $dismissed = get_user_meta(get_current_user_id(), 'real_estate_management_dismissable_notice', true);
    if ( !$dismissed) { ?>
        <div class="updated notice notice-success is-dismissible notice-get-started-class" data-notice="get_started" style="background: #f7f9f9; padding: 20px 10px; display: flex;">
	    	<div class="tm-admin-image">
	    		<img style="width: 100%;max-width: 320px;line-height: 40px;display: inline-block;vertical-align: top;border: 2px solid #ddd;border-radius: 4px;" src="<?php echo esc_url(get_stylesheet_directory_uri()) .'/screenshot.png'; ?>" />
	    	</div>
	    	<div class="tm-admin-content" style="padding-left: 30px; align-self: center">
	    		<h2 style="font-weight: 600;line-height: 1.3; margin: 0px;"><?php esc_html_e('Thank You For Choosing ', 'real-estate-management'); ?><?php echo wp_get_theme(); ?><h2>
	    		<p style="color: #3c434a; font-weight: 400; margin-bottom: 30px;"><?php _e('Get Started With Theme By Clicking On Getting Started.', 'real-estate-management'); ?><p>
	        	<a class="admin-notice-btn button button-primary button-hero" href="<?php echo esc_url( admin_url( 'themes.php?page=real-estate-management-info.php' )); ?>"><?php esc_html_e( 'Get started', 'real-estate-management' ) ?></a>
	        	<a class="admin-notice-btn button button-primary button-hero" target="_blank" href="<?php echo esc_url( REAL_ESTATE_MANAGEMENT_FREE_DOC ); ?>"><?php esc_html_e( 'Documentation', 'real-estate-management' ) ?></a>
	        	<span style="padding-top: 15px; display: inline-block; padding-left: 8px;">
	        	<span class="dashicons dashicons-admin-links"></span>
	        	<a class="admin-notice-btn"	 target="_blank" href="<?php echo esc_url( REAL_ESTATE_MANAGEMENT_LIVE_DEMO ); ?>"><?php esc_html_e( 'View Demo', 'real-estate-management' ) ?></a>
	        	</span>
	    	</div>
        </div>
    <?php }
}

add_action( 'admin_notices', 'real_estate_management_deprecated_hook_admin_notice' );

function real_estate_management_switch_theme() {
    delete_user_meta(get_current_user_id(), 'real_estate_management_dismissable_notice');
}
add_action('after_switch_theme', 'real_estate_management_switch_theme');
function real_estate_management_dismissable_notice() {
    update_user_meta(get_current_user_id(), 'real_estate_management_dismissable_notice', true);
    die();
}

/**
 * Theme Info.
 */
function real_estate_management_theme_info_load() {
	require get_theme_file_path( '/inc/theme-installation/theme-installation.php' );
}
add_action( 'init', 'real_estate_management_theme_info_load' );